## -*-Tcl-*-
 # ==========================================================================
 #  WWW Menu - an extension package for Alpha
 # 
 #  FILE: "wwwLinks.tcl"
 #                                    created: 04/30/1997 {11:04:46 am} 
 #                                last update: 12/04/2001 {18:20:10 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #     
 #  Description:
 #  
 #  Procedures to navigate, process links and the history cache.
 #  
 #  See the "wwwVersionHistory.tcl" file for license info, credits, etc.
 # ==========================================================================
 ##

# Make sure that the wwwMode.tcl file has been loaded.
wwwMode.tcl

proc wwwLinks.tcl {} {}

namespace eval WWW {}

# ===========================================================================
# 
#  ----  #
# 
#  Processing Links  #
# 

proc WWW::link {{to ""} args} {

    global mode WWW::FileSource WWW::Anchors mode WWWmodeVars
    
    # Make sure that the current window is added to the history
    # if necessary.
    if {$mode == "WWW"} {
	set title [win::Current]
	if {$title != "* WWW History *" && [info exists WWW::FileSource($title)]} {
	    set f WWW::FileSource($title)
	    WWW::addHistoryItem $f
	} 
    } 
    if {![string length $to]} {set to [lindex [WWW::getCurrentLink] 3]} 
    set to [string trim $to {\"}]
    set to [string trim $to]
    # We have to do this to in order to properly colorize visited links
    # when they are accessed via the mouse.
    if {[llength [winNames]]} {
	if {![isSelection]} {
	    if {![catch {WWW::getLinks} currentLinks]} {
		foreach linkList $currentLinks {
		    set link [lindex $linkList 3]
		    if {$link == $to} {
			select [lindex $linkList 0] [lindex $linkList 1]
			break
		    } 
		}
	    } elseif {[regexp {^[<:]:/[/]+} [getSelect]]} {
	        WWW::renderUrl [getSelect]
		return
	    }
	} 
	WWW::visitedLink $to
    }
    set l [string first ":" $to]
    switch [file extension $to] {
	".class" - 
	".java" {
	    set urlType "java"
	}
	default {
	    set urlType [string range $to 0 [expr $l -1]]
	}
    }
    if {[regsub {^\?+:[/]+} $to {} huh]} {
        status::errorMsg "Unknown server for '$huh'"
    } 
    # Do we have an anchor?
    if {[set anchorSpot [string last "\#" $to]] != -1} {
	set anchor [string range $to [expr {$anchorSpot + 1}] end]
	set to     [string range $to 0 [expr {$anchorSpot - 1}]]
    } else {
	set anchor ""
    }
    set urlType [string range $to 0 [expr $l -1]]
    global WWW::UrlAction
    if {[info exists WWW::UrlAction($urlType)]} {
	# do we handle this internally?
	global WWWmodeVars WWW::AlwaysInternal
	if {[info exists WWWmodeVars(${urlType}LinksInternal)]} {
	    set internal $WWWmodeVars(${urlType}LinksInternal)
	} else {
	    set internal 0
	}
	if {[lcontains WWW::AlwaysInternal $urlType] || $internal} {
	    # Do we replace the current window?
	    [set WWW::UrlAction($urlType)] $to $anchor
	    return
	}
    }
    # if we didn't return above
    WWW::externalLink $to
}

proc WWW::fileLink {to args} {
    
    global ModeSuffixes WWW::Anchors

    WWW::massagePath to
    if {[case [file extension $to] $ModeSuffixes] != "HTML"} {
	file::openQuietly $to
	return
    } else {
	WWW::renderFile $to
    }
    if {[llength args] && [info exists WWW::Anchors([win::Current])]} {
	eval set anchor $args
	foreach Anchor [set WWW::Anchors([win::Current])] {
	    if {$anchor == [lindex $Anchor 0]} {
		goto [lindex $Anchor 1] ; insertToTop
		return
	    } 
	}
    }
}

proc WWW::ftpLink {to args} {

    regsub {^ftp:[/]+} $to {} to
    url::parseFtp $to i
    if {![catch {ftp::browse $i(host) $i(path) $i(user) $i(pass) $i(file)}]} {
	set alert    "This file will be opened in Alpha, but in a temp cache -- "
	append alert "You can use 'WWW --> Save Source As' to save it on a local disk."
	alertnote $alert
    }
    return
    
    # This doesn't work because the file might not have landed yet.
    
    if {![catch {ftp::browse $i(host) $i(path) $i(user) $i(pass) $i(file)}]} {
	set question    "This file was opened in Alpha.\r"
	append question "Would you prefer to save it on a local disk?"
	if {[askyesno $question] == "yes"} {
	    menu::fileProc {File} {saveAs}
	    killWindow
	} 
    }
}

proc WWW::httpLink {to args} {

    WWW::renderUrl $to ""
    if {[llength args] && [info exists WWW::Anchors([win::Current])]} {
	eval set anchor $args
	foreach Anchor [set WWW::Anchors([win::Current])] {
	    if {$anchor == [lindex $Anchor 0]} {
		goto [lindex $Anchor 1] ; insertToTop
		return
	    } 
	}
    }
}

proc WWW::javaLink {to args} {

    global WWWmodeVars 
    
    WWW::massagePath to
    if {$WWWmodeVars(runJavaAppletsDirectly)} {
	# can run applet directly
	alertnote "Sorry, I don't yet know how to run .class files directly."
	javaRun "[file root ${to}].class"
    } else {
	# use html file
	global javaViewerSig WWW::Pages WWW::PagePos
	set app [file tail [app::launchFore $javaViewerSig]]
	sendOpenEvent -n $app [lindex [lindex [set WWW::Pages] [set WWW::PagePos]] 0]
    }
}

proc WWW::externalLink {to args} {
    
    global WWWmodeVars
    
    if {$WWWmodeVars(wwwSendRemoteLinks)} {
	url::execute $to
    } else {
	set    alert "External link to \r'$to',\r"
	append alert "toggle the WWW mode flags to use a helper application, "
	append alert "or to pass on any 'unknown' remote links "
	append alert "to Internet Config."
	alertnote $alert
	refresh
    }
}

# ===========================================================================
# 
#  ----  #
# 
#  Managing Links  #
# 

proc WWW::visitedLink {to} {
    
    global WWW::Visited WWW::Links
    
    if {![llength [winNames]]} {return}
    
    set to [string trim $to {\"}]
    if {![lcontains [set WWW::Visited] $to]} {lappend WWW::Visited $to}
    if {[isSelection] && ![catch {WWW::getCurrentLink} result]} {
	set pos0 [lindex $result 0]
	set pos1 [lindex $result 1]
	set link [lindex $result 3]
	WWW::makeLink [win::Current] $pos0 $pos1 $link
    } 
}

proc WWW::nextLink {direction {pos ""} {title ""}} {
    
    global WWWmodeVars wwwMenu
    
    if {![string length $pos]} {
	if {$direction} {
	    set pos [pos::math [selEnd] - 1]
	} else {
	    set pos [getPos]
	}
	# Make sure that the cursor is somewhere in the window.
	getWinInfo winArray
	set top   $winArray(currline)
	set lines $winArray(linesdisp)
	# This is the bottom of the window.
	set pos0  [nextLineStart [rowColToPos [expr {$top + $lines}] 0]]
	# This is the top of the window.
	set pos1  [pos::math [nextLineStart [rowColToPos $top 0]] - 1]
	if {[pos::compare $pos > $pos0] || [pos::compare $pos < $pos1]} {
	    set pos [set pos${direction}]
	}
    }

    # Now find the closest link.
    if {![catch {WWW::findLink $direction $pos $title} result]} {
	set pos0 [lindex $result 0]
	set pos1 [lindex $result 1]
	set cmd  [lindex $result 2]
	set link [lindex $result 3]
	select $pos0 $pos1
	if {$WWWmodeVars(centerRefreshOnNav)} {centerRedraw}
	WWW::postEvalLinks
	status::msg $cmd
    } else {
	WWW::postEvalLinks
	status::errorMsg "Couldn't find any links in this window."
    }
}

proc WWW::findLink {direction pos {title ""}} {
    
    foreach linkList [WWW::getLinks $title $direction] {
	set pos0 [lindex $linkList 0]
	if {$direction && [pos::compare $pos >= $pos0]} {
	    # Keep looking for the next link below this one.
	    continue
	} elseif {!$direction && [pos::compare $pos <= $pos0]} {
	    # Keep looking for the next link above this one.
	    continue
	} else {
	    set result $linkList
	    break
	}
    }
    if {[info exists result]} {
	return $result
    } else {
	# We reached the end/beginning of the window.
	beep
	if {$direction} {
	    return [WWW::findLink $direction [minPos] $title]
	} else {
	    return [WWW::findLink $direction [maxPos] $title]
	}
    }
}

proc WWW::getLinks {{title ""} {direction 1}} {

    global mode WWW::Links WWW::LinksReverse
    
    if {$mode != "WWW"} {error "links are only found in WWW windows."}
    # Don't want 'tail' since these are not file windows and may
    # contain strange characters (:/ etc) in the name
    if {![string length $title]} {set title [win::Current]}
    if {![info exists WWW::Links($title)] || ![llength [set WWW::Links($title)]]} {
	status::errorMsg "No links found in '$title'"
    } elseif {![info exists WWW::LinksReverse($title)]} {
	# Create the reverse lookup if it doesn't exist.
	set WWW::LinksReverse($title) [lreverse [set WWW::Links($title)]]
    } 
    if {$direction} {
	return [set WWW::Links($title)]
    } else {
	return [set WWW::LinksReverse($title)]
    }
}

proc WWW::getCurrentLink {{title ""} {quietly 1}} {

    global WWW::Links

    if {![string length $title]} {set title [win::Current]}
    set links [WWW::getLinks $title]
    set pos0 [getPos]
    set pos1 [selEnd]
    foreach link [set WWW::Links($title)] {
	set pos2 [lindex $link 0]
	set pos3 [lindex $link 1]
	if {[pos::compare $pos0 >= $pos2] && [pos::compare $pos1 <= $pos3]} {
	    set result $link
	    break
	} else {
	    continue
	}
    }
    if {[info exists result]} {
	if {!$quietly} {status::msg "Links to [lindex $result 3]"}
	return $result
    } else {
	if {!$quietly} {status::errorMsg "The current selection is not a link."}
	error "The current selection is not a link."
    }
}

proc WWW::modifyLink {} {
    
    global mode WWW::Pages WWW::PagePos WWW::Links WWW::LinksReverse
    
    # Make sure that we have enough info to do this.
    if {$mode != "WWW"} {
	dialog::errorAlert "Only useful in WWW browser mode."
    }
    set title [win::Current]
    if {[catch {set WWW::FileSource($title)} fileSource]} {
        dialog::errorAlert "Sorry, can't identify the source for '$title'."
    } elseif {![file exists $fileSource]} {
        dialog::errorAlert "Sorry, can't find '$fileSource'."
    }
    # Find the current link to modify.
    set linkList [WWW::getCurrentLink]
    set oldLink  [lindex $linkList 3]
    if {[catch {dialog::getUrl "Enter new link location" $oldLink} to]} {
	status::errorMsg "Cancelled."
    } elseif {$to == "" || $to == $oldLink} {
	status::msg "Nothing was changed."
	return
    }
    # Find out if the source is already open.
    set w [win::Current]
    if {![catch {getWinInfo -w $fileSource i}]} {
	if {$i(dirty)} {
	    if {![dialog::yesno "Save original file?"]} {
		status::errorMsg "Cancelled."
	    } 
	    status::msg "Saving original file."
	    bringToFront $fileSource
	    save
	    bringToFront $w
	}
    }
    # Update the source file.
    set reglink [quote::Regfind $oldLink]
    set regto   [quote::Regsub $to]
    set cid [alphaOpen $fileSource "r"]
    if {[regsub -all -- $reglink [read $cid] $regto out]} {
	set ocid [alphaOpen $fileSource "w+"]
	puts -nonewline $ocid $out
	close $ocid
	status::msg "Updated original."
    }
    close $cid
    if {![catch {bringToFront $fileSource}]} {
	status::msg "Updating window to agree with disk version."
	revert
	bringToFront $w
    }
    setWinInfo read-only 0	
    WWW::makeLink [win::Current] [getPos] [selEnd] $to
    setWinInfo read-only 1
    # Now update the link lists.
    set i1 [lsearch [set WWW::Links($w)] $linkList]
    set i2 [lsearch [set WWW::LinksReverse($w)] $linkList]
    set linkList [lreplace $linkList 2 2 "WWW::link \"$to\""] 
    set linkList [lreplace $linkList 3 3 $to]
    set WWW::Links($w)        [lreplace [set WWW::Links($w)]        $i1 $i1 $linkList]
    set WWW::LinksReverse($w) [lreplace [set WWW::LinksReverse($w)] $i2 $i2 $linkList]
    status::msg "WWW::link \"$to\""

}

proc WWW::displayLinks {} {
    
    global WWW::Links
    
    set title [win::Current]
    set links [WWW::getLinks]

    # Create the introduction.
    set    intro "\rThis window contains all of the hyperlinks found in the rendered\r"
    append intro "\"$title\" window.\r\r"
    append intro "Click here: <<WWW::hyperlinkLinks>> to hyperlink this window.\r"
    append intro "\r__________________________________________________________________________\r\r"
    set newTitle "* [win::MakeTitle "Links in '$title"]' *"
    foreach window [winNames] {
	if {[win::StripCount $window] == $newTitle} {
	    bringToFront $window ; killWindow
	} 
    }
    # Add the links in the current window.
    set count   0
    set results ""
    foreach linkList $links {
	set pos0 [lindex $linkList 0]
	set pos1 [lindex $linkList 1]
	regsub -all "\[\r\n\t\]+" [getText $pos0 $pos1] { } name
	append results "\t  \t\"$name\" \r    <[lindex $linkList 3]>\r\r"
	incr count
    }
    # Create the new window, color and hyperize, remove quotes, and mark it.
    new -n $newTitle -m "Text" -text ${intro}${results} -tabsize 1
    win::searchAndHyperise {<<([^>\r\n]+)>>}  {\1} 1 4 +2 -2
    win::searchAndHyperise {\"([^\r\n\"]+)\"} {}   1 1 +1 -1
    refresh
    set pos [minPos]
    while {![catch {search -s -f 1 -r 1 {\"} $pos} match]} {
	set pos [lineStart [lindex $match 1]]
	replaceText [lindex $match 0] [lindex $match 1] ""
    }
    markFile ; sortMarksFile
    winReadOnly
    status::msg "'$title' contains $count hyperlinks."
}

proc WWW::hyperlinkLinks {} {
    
    win::searchAndHyperise \
      {<([^:]+:/[/]+[^ >]*)>} \
      {url::execute "\1"} 1
    win::searchAndHyperise \
      {<(mailto:[-_a-zA-Z0-9.]+@[-_a-zA-Z0-9.]+)>} \
      {url::execute "\1"} 1
    refresh
}

# ===========================================================================
# 
#  ----  #
# 
#  Managing History Cache  #
# 

proc WWW::history {} {

    global WWW::Links WWW::LinksReverse WWW::History WWW::Marks
    
    set title "* WWW History *"
    catch {unset  WWW::Links($title)}
    catch {unset  WWW::LinksReverse($title)}
    if {[llength [set WWW::History]] == "1"} {
	status::msg "There are no items in the history cache."
	return
    } 
    foreach window [winNames] {
	if {[win::StripCount $window] == $title} {
	    bringToFront $window ; killWindow
	} 
    }
    set count 0
    set WWW::Marks($title) [list ]
    # This history will contain a lot of duplicates.  We'll
    # reverse the order (so that the most recent entries appear
    # first), then make the list unique.  While we're at it,
    # we'll save this unique list as the history so that it is
    # a little easier the next time that this is called.
    set WWW::History [lreverse [lunique [lreverse [set WWW::History]]]]
    set thisDate [mtime [now] short]
    set newHistory [set histDate [lindex [set WWW::History] 0]]
    append intro "\rCurrent date and time:   ${thisDate}"
    append intro "\rWWW pages rendered since ${histDate}\r"
    append intro "\rTo remove individual history entries click here: <<WWW::deleteHistoryItem>>,"
    append intro "\ror use this link <<WWW::clearHistory>> to clear the entire list.\r"
    append intro "\r__________________________________________________________________________\r\r"
    new -n $title -m "WWW" -text $intro
    foreach page [lreverse [lrange [set WWW::History] 1 end]] {
	set name [lindex $page 0]
	set pos0 [getPos]
	insertText "\"$name\" --\r    "
	set pos1 [getPos]
	insertText [set link [lindex $page 1]]
	set pos2 [getPos] ; insertText \r\r
	set cmd [WWW::makeLink [win::Current] $pos1 $pos2 $link]
	lappend WWW::Links($title) [list $pos1 $pos2 $cmd $link]
	lappend reversedPages $page
	set name [win::MakeTitle $name]
	while {[lcontains marks $name]} {append name " "}
	lappend marks $name
	setNamedMark $name $pos0 $pos0 $pos0
	lappend WWW::Marks($title) [list $name $pos0 $pos0 $pos0]
	incr count
    }
    set WWW::History [concat [list $histDate] [lreverse $reversedPages]]
    set WWW::Marks($title) [lsort -ignore [set WWW::Marks($title)]]
    sortMarksFile
    win::searchAndHyperise {<<([^>\r\n]+)>>} {\1} 1 4 +2 -2
    refresh
    winReadOnly
    status::msg "The history cache currently includes $count pages."
}

proc WWW::addHistoryItem {f {title ""}} {
    
    global WWW::Pages WWW::PagePos WWW::BaseUrl WWW::History
    
    if {![string length $title]} {set title [win::Current]}
    if {$f == "* WWW History *" || $title == "* WWW History *"} {
	return
    }
    # First add it to the 'Go To Page' menu.
    if {[set i [lsearch -glob [set WWW::Pages] [list * [win::Current]]]] != -1} {
	set WWW::PagePos $i
    } else {		
	set  WWW::Pages [lrange [set WWW::Pages] 0 [set WWW::PagePos]]
	incr WWW::PagePos
	lappend WWW::Pages [list $f [win::Current]]
    }
    # Now add it to the History.
    set title [string trim $title]
    if {[info exists WWW::BaseUrl($title)]} {
	lappend WWW::History [list $title [set WWW::BaseUrl($title)]]
    } 
    menu::buildSome goToPage
}

proc WWW::deleteHistoryItem {} {
    
    global WWW::History

    set p "Remove which history items?"
    set items [lsort -ignore [lreverse [lunique [lrange [set WWW::History] 1 end]]]]
    if {[catch {listpick -l -p $p $items} result] || ![llength $result]} {
	status::errorMsg "Cancelled."
    } 
    foreach item $result {
	set WWW::History [lremove [set WWW::History] $item]
    }
    set reopen 0
    foreach window [winNames] {
	if {[win::StripCount $window] == "* WWW History *"} {
	    bringToFront $window ; killWindow
	    set reopen 1
	} 
    }
    if {$reopen} {WWW::menuProc "" "history"}
    status::msg "Selected items have been removed."
}

proc WWW::clearHistory {} {
    
    global WWW::History

    foreach window [winNames] {
	if {[win::StripCount $window] == "* WWW History *"} {
	    bringToFront $window ; killWindow
	} 
    }
    set WWW::History [list [mtime [now] short]]
    WWW::postEval
    status::msg "The WWW history has been flushed."
}

# ===========================================================================
# 
# .